<?php

/**
* The public-facing functionality of the plugin.
*
* @link       https://wpdeveloper.net
* @since      1.0.0
*
* @package    NotificationX
* @subpackage NotificationX/public
*/

/**
* The public-facing functionality of the plugin.
*
* Defines the plugin name, version, and two examples hooks for how to
* enqueue the public-facing stylesheet and JavaScript.
*
* @package    NotificationX
* @subpackage NotificationX/public
* @author     WPDeveloper <support@wpdeveloper.net>
*/
class NotificationX_Public {
	
	/**
	* The ID of this plugin.
	*
	* @since    1.0.0
	* @access   private
	* @var      string    $plugin_name    The ID of this plugin.
	*/
	private $plugin_name;
	
	/**
	* The version of this plugin.
	*
	* @since    1.0.0
	* @access   private
	* @var      string    $version    The current version of this plugin.
	*/
	private $version;
	
	public static $active = [];
	
	public $notifications = [];
	
	/**
	* Initialize the class and set its properties.
	*
	* @since    1.0.0
	* @param      string    $plugin_name       The name of the plugin.
	* @param      string    $version    The version of this plugin.
	*/
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name   = $plugin_name;
		$this->version       = $version;
		
		$this->notifications = get_option('notificationx_data');

		add_filter('body_class', array($this,'body_class'), 10, 1);
	}
	
	/**
	* Register the stylesheets for the public-facing side of the site.
	*
	* @since    1.0.0
	*/
	public function enqueue_styles() {
		
		/**
		* This function is provided for demonstration purposes only.
		*
		* An instance of this class should be passed to the run() function
		* defined in NotificationX_Loader as all of the hooks are defined
		* in that particular class.
		*
		* The NotificationX_Loader will then create the relationship
		* between the defined hooks and the functions defined in this
		* class.
		*/
		
		wp_enqueue_style( $this->plugin_name, NOTIFICATIONX_PUBLIC_URL . 'assets/css/notificationx-public.min.css', array(), $this->version, 'all' );
	}
	
	/**
	* Register the JavaScript for the public-facing side of the site.
	*
	* @since    1.0.0
	*/
	public function enqueue_scripts() {
		
		/**
		* This function is provided for demonstration purposes only.
		*
		* An instance of this class should be passed to the run() function
		* defined in NotificationX_Loader as all of the hooks are defined
		* in that particular class.
		*
		* The NotificationX_Loader will then create the relationship
		* between the defined hooks and the functions defined in this
		* class.
		*/
		
		wp_enqueue_script( $this->plugin_name . '-cookie', NOTIFICATIONX_PUBLIC_URL . 'assets/js/Cookies.js', array( 'jquery' ), $this->version, true );
		wp_enqueue_script( $this->plugin_name, NOTIFICATIONX_PUBLIC_URL . 'assets/js/notificationx-public.min.js', array( 'jquery' ), $this->version, true );
	}
	
	/**
	* Get all active notifications.
	*
	* @since 1.0.0
	* @return void
	*/
	public static function get_active_items() {
		$args = array(
			'post_type'         => 'notificationx',
			'posts_per_page'    => '-1',
			'post_status'		=> 'publish',
			'meta_query'        => array(
				array(
					'key'           => '_nx_meta_active_check',
					'value'         => '1',
					'compare'       => '='
				)
			)
		);
		$posts = get_posts( $args );
		if ( ! empty( $posts ) ) {
			foreach ( $posts as $post ) {
				self::$active[] = $post->ID;
			}
		}

		return self::$active;
	}

	public function pro_extension_ids() {
		return apply_filters('nx_pro_extetion_ids', array());
	}

	public function generate_active_notificationx(){
		
		if( empty( self::$active ) ) {
			return;
		}
		$activeItems = self::$active;
		$conversion_ids = $comments_id = $reviews_id = $download_stats_id = array();
		
		foreach( self::$active as $id ) {
			
			$settings = NotificationX_MetaBox::get_metabox_settings( $id );
			self::generate_css( $settings );
			
			$logged_in = is_user_logged_in();
			$show_on_display = $settings->show_on_display;
			
			if( ( $logged_in && 'logged_out_user' == $show_on_display ) || ( ! $logged_in && 'logged_in_user' == $show_on_display ) ) {
				continue;
			}
			
			$locations = $settings->all_locations;
			
			$check_location = false;
			
			if( ! empty( $locations ) ) {
				$check_location = NotificationX_Locations::check_location( array( $locations ) );
			}
			$check_location = apply_filters( 'nx_check_location', $check_location, $settings );
			
			if( $settings->show_on == 'on_selected' ) {
				// show if the page is on selected
				if ( ! $check_location ) {
					continue;
				}
			} elseif( $settings->show_on == 'hide_on_selected' ) {
				// hide if the page is on selected
				if ( $check_location ) {
					continue;
				}
			}
			/**
			* Check for hiding in mobile device
			*/
			if( wp_is_mobile() && $settings->hide_on_mobile ) {
				continue;
			}
			
			$add_in_list = apply_filters( 'nx_add_in_queue', $settings->display_type, $settings );

			switch ( $add_in_list  ) {
				case "press_bar":
					NotificationX_PressBar_Extension::display( $settings );
					break;
				case "conversions":
					$conversion_ids[] = $id;
					break;
				case "comments":
					$comments_id[] = $id;
					break;
				case "reviews":
					$reviews_id[] = $id;
					break;
				case "download_stats":
					$download_stats_id[] = $id;
					break;
			}
			unset( $activeItems[ $id ] );
		}
		do_action( 'nx_active_notificationx', $activeItems );
		$pro_ext = $this->pro_extension_ids();
		/**
		* Filtered Active IDs
		*/
		$conversion_ids              = apply_filters('nx_conversions_id', $conversion_ids );
		$comments_id                 = apply_filters('nx_comments_id', $comments_id );
		$reviews_id                  = apply_filters('nx_reviews_id', $reviews_id );
		$reviedownload_stats_idws_id = apply_filters('nx_download_stats_id', $download_stats_id );
		
		if( ! empty( $conversion_ids ) || ! empty( $comments_id ) || ! empty( $pro_ext ) || ! empty( $reviews_id ) || ! empty( $download_stats_id ) ) :
			?>
			<script type="text/javascript">
			var notificationx = {
				nonce      : '<?php echo wp_create_nonce('nx_frontend_nonce'); ?>',
				ajaxurl    : '<?php echo admin_url('admin-ajax.php'); ?>',
				conversions: <?php echo json_encode( $conversion_ids ); ?>,
				comments   : <?php echo json_encode( $comments_id ); ?>,
				reviews   : <?php echo json_encode( $reviews_id ); ?>,
				stats   : <?php echo json_encode( $download_stats_id ); ?>,
				pro_ext   : <?php echo json_encode( $pro_ext ); ?>,
			};
			</script>
			<?php	
		endif;
	}
			
	public function generate_conversions() {
		
		if( ! isset( $_POST['nonce'] ) && ! wp_verify_nonce( $_POST['nonce'], 'nx_frontend_nonce' ) ) {
			return;
		}
		
		$ids = $_POST['ids'];
		
		$echo = $data = [];
		if( ! empty( $this->notifications ) ) {
			$data = $this->notifications;
		}
		
		$settings = NotificationX_MetaBox::get_metabox_settings( $ids );
		$echo['config'] = apply_filters('nx_frontend_config', array(
			'delay_before'  => ( ! empty( $settings->delay_before ) ) ? intval( $settings->delay_before ) * 1000 : 0,
			'display_for'   => ( ! empty( $settings->display_for ) ) ? intval( $settings->display_for ) * 1000 : 0,
			'delay_between' => ( ! empty( $settings->delay_between ) ) ? intval( $settings->delay_between ) * 1000 : 0,
			'loop'          => ( ! empty( $settings->loop ) ) ? $settings->loop : 0,
			'id'            => $ids,
		), $settings);

		ob_start();
		include NOTIFICATIONX_PUBLIC_PATH . 'partials/nx-public-display.php';
		$content = ob_get_clean();
		$echo['content'] = $content;
		
		echo json_encode( $echo );
		wp_die();
	}
			
	public function get_client_ip() {
		$ip = '';
		
		if( getenv( 'HTTP_CLIENT_IP' ) ) {
			$ip = getenv( 'HTTP_CLIENT_IP' );
		} elseif( getenv( 'HTTP_X_FORWARDED_FOR' ) ) {
			$ip = getenv( 'HTTP_X_FORWARDED_FOR' );
		} elseif( getenv( 'HTTP_X_FORWARDED' ) ) {
			$ip = getenv( 'HTTP_X_FORWARDED' );
		} elseif( getenv( 'HTTP_FORWARDED_FOR' ) ) {
			$ip = getenv( 'HTTP_FORWARDED_FOR' );
		} elseif( getenv( 'HTTP_FORWARDED' ) ) {
			$ip = getenv( 'HTTP_FORWARDED' );
		} elseif( getenv( 'REMOTE_ADDR' ) ) {
			$ip = getenv( 'REMOTE_ADDR' );
		} else {
			$ip = "UNKNOWN";
		}
		
		return $ip;
	}
			
	public static function generate_css( $settings ){
		echo NotificationX_Advanced_Style::generate_css( $settings );
	}
	/**
	* This function is responsible for generate css for preview
	* 
	* @param stdClass $settings
	* @param string $key
	* @return string
	*/
	public static function generate_preview_css( $settings, $key = 'wrapper' ){
		if( empty( $settings ) ) return;
		$style = $image_style = $content_style = $first_row_font = $second_row_font = $third_row_font = [];
		$css_string = $css = '';
		
		switch( $settings->display_type ) {
			case 'conversions' : 
				if( $settings->advance_edit ) {
					$style[ 'wrapper' ][] = ! empty( $settings->bg_color ) ? 'background-color: ' . $settings->bg_color : '';
					$style[ 'wrapper' ][] = ! empty( $settings->text_color ) ? 'color: ' . $settings->text_color : '';
					
					if( $settings->border ){
						$style[ 'wrapper' ][] = ! empty( $settings->border_size ) ? 'border-width: ' . $settings->border_size . 'px !important' : '';
						$style[ 'wrapper' ][] = ! empty( $settings->border_style ) ? 'border-style: ' . $settings->border_style . ' !important': '';
						$style[ 'wrapper' ][] = ! empty( $settings->border_color ) ? 'border-color: ' . $settings->border_color  . ' !important': '';
					}
					
					if( ! empty( $settings->first_font_size ) ) {
						$style['first-row'][] = 'font-size: ' . $settings->first_font_size . 'px';
					}
					if( ! empty( $settings->second_font_size ) ) {
						$style['second-row'][] = 'font-size: ' . $settings->second_font_size . 'px';
					}
					if( ! empty( $settings->third_font_size ) ) {
						$style['third-row'][] = 'font-size: ' . $settings->third_font_size . 'px';
					}
					
					return 'style="'. implode( '; ', $style[ $key ] ) .'"';
				}
				break;
			case 'comments' : 
				if( $settings->comment_advance_edit ) {
					$style[ 'wrapper' ][] = ! empty( $settings->comment_bg_color ) ? 'background-color: ' . $settings->comment_bg_color : '';
					$style[ 'wrapper' ][] = ! empty( $settings->comment_text_color ) ? 'color: ' . $settings->comment_text_color : '';
					
					if( $settings->comment_border ){
						$style[ 'wrapper' ][] = ! empty( $settings->comment_border_size ) ? 'border-width: ' . $settings->comment_border_size . 'px !important' : '';
						$style[ 'wrapper' ][] = ! empty( $settings->comment_border_style ) ? 'border-style: ' . $settings->comment_border_style . ' !important' : '';
						$style[ 'wrapper' ][] = ! empty( $settings->comment_border_color ) ? 'border-color: ' . $settings->comment_border_color . ' !important' : '';
					}
					
					if( ! empty( $settings->comment_first_font_size ) ) {
						$style['first-row'][] = 'font-size: ' . $settings->comment_first_font_size . 'px';
					}
					if( ! empty( $settings->comment_second_font_size ) ) {
						$style['second-row'][] = 'font-size: ' . $settings->comment_second_font_size . 'px';
					}
					if( ! empty( $settings->comment_third_font_size ) ) {
						$style['third-row'][] = 'font-size: ' . $settings->comment_third_font_size . 'px';
					}
					return 'style="'. implode( '; ', $style[ $key ] ) .'"';
				}
				break;
			default : 
				return '';
				break;
		}
	}
	/**
	* This function is responsible for generate css for preview
	* its use when advance edit button is clicked.
	* @param stdClass $settings
	* @param string $key
	* @return string
	*/
	public static function generate_css_for_preview( $settings, $key = 'wrapper' ){
		if( empty( $settings ) ) return;
		$style = $image_style = $content_style = $first_row_font = $second_row_font = $third_row_font = [];
		$css_string = $css = '';
		
		switch( $settings->display_type ) {
			case 'conversions' : 
				$style[ 'wrapper' ][] = ! empty( $settings->bg_color ) ? 'background-color: ' . $settings->bg_color : '';
				$style[ 'wrapper' ][] = ! empty( $settings->text_color ) ? 'color: ' . $settings->text_color : '';
				
				if( $settings->border ){
					$style[ 'wrapper' ][] = ! empty( $settings->border_size ) ? 'border-width: ' . $settings->border_size . 'px' : '';
					$style[ 'wrapper' ][] = ! empty( $settings->border_style ) ? 'border-style: ' . $settings->border_style : '';
					$style[ 'wrapper' ][] = ! empty( $settings->border_color ) ? 'border-color: ' . $settings->border_color : '';
				}
				
				if( ! empty( $settings->first_font_size ) ) {
					$style['first-row'][] = 'font-size: ' . $settings->first_font_size . 'px';
				}
				if( ! empty( $settings->second_font_size ) ) {
					$style['second-row'][] = 'font-size: ' . $settings->second_font_size . 'px';
				}
				if( ! empty( $settings->third_font_size ) ) {
					$style['third-row'][] = 'font-size: ' . $settings->third_font_size . 'px';
				}
				return implode( ';', $style[ $key ] );
				break;
			case 'comments' : 
				$style[ 'wrapper' ][] = ! empty( $settings->comment_bg_color ) ? 'background-color: ' . $settings->comment_bg_color : '';
				$style[ 'wrapper' ][] = ! empty( $settings->comment_text_color ) ? 'color: ' . $settings->comment_text_color : '';
				
				if( $settings->comment_border ){
					$style[ 'wrapper' ][] = ! empty( $settings->comment_border_size ) ? 'border-width: ' . $settings->comment_border_size . 'px' : '';
					$style[ 'wrapper' ][] = ! empty( $settings->comment_border_style ) ? 'border-style: ' . $settings->comment_border_style : '';
					$style[ 'wrapper' ][] = ! empty( $settings->comment_border_color ) ? 'border-color: ' . $settings->comment_border_color : '';
				}
				
				if( ! empty( $settings->comment_first_font_size ) ) {
					$style['first-row'][] = 'font-size: ' . $settings->comment_first_font_size . 'px';
				}
				if( ! empty( $settings->comment_second_font_size ) ) {
					$style['second-row'][] = 'font-size: ' . $settings->comment_second_font_size . 'px';
				}
				if( ! empty( $settings->comment_third_font_size ) ) {
					$style['third-row'][] = 'font-size: ' . $settings->comment_third_font_size . 'px';
				}
				return implode( ';', $style[ $key ] );
				break;
			default : 
				return '';
				break;
		}
	}
    /**
     * This function added css class in body for bar theme
     * @hooked body_class
     * @param array $classes
     * @return array
     */
    public function body_class($classes){

        if(!empty(self::$active)){
            foreach (self::$active as $active_nx){
                $type = get_post_meta($active_nx, '_nx_meta_display_type', true);
                if($type == 'press_bar'){
                    $classes[] = 'nx-'. $type . '-active';
                    $classes[] = 'nx-'. $type . '-active-' . $active_nx;
                }
            }
        }

	    return $classes;
    }
}
