<?php

namespace MailPoetVendor\Sabberworm\CSS\CSSList;

use MailPoetVendor\Sabberworm\CSS\Renderable;
use MailPoetVendor\Sabberworm\CSS\RuleSet\DeclarationBlock;
use MailPoetVendor\Sabberworm\CSS\RuleSet\RuleSet;
use MailPoetVendor\Sabberworm\CSS\Property\Selector;
use MailPoetVendor\Sabberworm\CSS\Comment\Commentable;
/**
 * A CSSList is the most generic container available. Its contents include RuleSet as well as other CSSList objects.
 * Also, it may contain Import and Charset objects stemming from @-rules.
 */
abstract class CSSList implements \MailPoetVendor\Sabberworm\CSS\Renderable, \MailPoetVendor\Sabberworm\CSS\Comment\Commentable
{
    protected $aComments;
    protected $aContents;
    protected $iLineNo;
    public function __construct($iLineNo = 0)
    {
        $this->aComments = array();
        $this->aContents = array();
        $this->iLineNo = $iLineNo;
    }
    /**
     * @return int
     */
    public function getLineNo()
    {
        return $this->iLineNo;
    }
    public function append($oItem)
    {
        $this->aContents[] = $oItem;
    }
    /**
     * Removes an item from the CSS list.
     * @param RuleSet|Import|Charset|CSSList $oItemToRemove May be a RuleSet (most likely a DeclarationBlock), a Import, a Charset or another CSSList (most likely a MediaQuery)
     */
    public function remove($oItemToRemove)
    {
        $iKey = \array_search($oItemToRemove, $this->aContents, \true);
        if ($iKey !== \false) {
            unset($this->aContents[$iKey]);
            return \true;
        }
        return \false;
    }
    /**
     * Set the contents.
     * @param array $aContents Objects to set as content.
     */
    public function setContents(array $aContents)
    {
        $this->aContents = array();
        foreach ($aContents as $content) {
            $this->append($content);
        }
    }
    /**
     * Removes a declaration block from the CSS list if it matches all given selectors.
     * @param array|string $mSelector The selectors to match.
     * @param boolean $bRemoveAll Whether to stop at the first declaration block found or remove all blocks
     */
    public function removeDeclarationBlockBySelector($mSelector, $bRemoveAll = \false)
    {
        if ($mSelector instanceof \MailPoetVendor\Sabberworm\CSS\RuleSet\DeclarationBlock) {
            $mSelector = $mSelector->getSelectors();
        }
        if (!\is_array($mSelector)) {
            $mSelector = \explode(',', $mSelector);
        }
        foreach ($mSelector as $iKey => &$mSel) {
            if (!$mSel instanceof \MailPoetVendor\Sabberworm\CSS\Property\Selector) {
                $mSel = new \MailPoetVendor\Sabberworm\CSS\Property\Selector($mSel);
            }
        }
        foreach ($this->aContents as $iKey => $mItem) {
            if (!$mItem instanceof \MailPoetVendor\Sabberworm\CSS\RuleSet\DeclarationBlock) {
                continue;
            }
            if ($mItem->getSelectors() == $mSelector) {
                unset($this->aContents[$iKey]);
                if (!$bRemoveAll) {
                    return;
                }
            }
        }
    }
    public function __toString()
    {
        return $this->render(new \MailPoetVendor\Sabberworm\CSS\OutputFormat());
    }
    public function render(\MailPoetVendor\Sabberworm\CSS\OutputFormat $oOutputFormat)
    {
        $sResult = '';
        $bIsFirst = \true;
        $oNextLevel = $oOutputFormat;
        if (!$this->isRootList()) {
            $oNextLevel = $oOutputFormat->nextLevel();
        }
        foreach ($this->aContents as $oContent) {
            $sRendered = $oOutputFormat->safely(function () use($oNextLevel, $oContent) {
                return $oContent->render($oNextLevel);
            });
            if ($sRendered === null) {
                continue;
            }
            if ($bIsFirst) {
                $bIsFirst = \false;
                $sResult .= $oNextLevel->spaceBeforeBlocks();
            } else {
                $sResult .= $oNextLevel->spaceBetweenBlocks();
            }
            $sResult .= $sRendered;
        }
        if (!$bIsFirst) {
            // Had some output
            $sResult .= $oOutputFormat->spaceAfterBlocks();
        }
        return $sResult;
    }
    /**
     * Return true if the list can not be further outdented. Only important when rendering.
     */
    public abstract function isRootList();
    public function getContents()
    {
        return $this->aContents;
    }
    /**
     * @param array $aComments Array of comments.
     */
    public function addComments(array $aComments)
    {
        $this->aComments = \array_merge($this->aComments, $aComments);
    }
    /**
     * @return array
     */
    public function getComments()
    {
        return $this->aComments;
    }
    /**
     * @param array $aComments Array containing Comment objects.
     */
    public function setComments(array $aComments)
    {
        $this->aComments = $aComments;
    }
}
