<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\ORM\Mapping;

/**
 * Naming strategy implementing the underscore naming convention.
 * Converts 'MyEntity' to 'my_entity' or 'MY_ENTITY'.
 *
 * 
 * @link    www.doctrine-project.org
 * @since   2.3
 * @author  Fabio B. Silva <fabio.bat.silva@gmail.com>
 */
class UnderscoreNamingStrategy implements \MailPoetVendor\Doctrine\ORM\Mapping\NamingStrategy
{
    /**
     * @var integer
     */
    private $case;
    /**
     * Underscore naming strategy construct.
     *
     * @param integer $case CASE_LOWER | CASE_UPPER
     */
    public function __construct($case = \CASE_LOWER)
    {
        $this->case = $case;
    }
    /**
     * @return integer CASE_LOWER | CASE_UPPER
     */
    public function getCase()
    {
        return $this->case;
    }
    /**
     * Sets string case CASE_LOWER | CASE_UPPER.
     * Alphabetic characters converted to lowercase or uppercase.
     * 
     * @param integer $case
     *
     * @return void
     */
    public function setCase($case)
    {
        $this->case = $case;
    }
    /**
     * {@inheritdoc}
     */
    public function classToTableName($className)
    {
        if (\strpos($className, '\\') !== \false) {
            $className = \substr($className, \strrpos($className, '\\') + 1);
        }
        return $this->underscore($className);
    }
    /**
     * {@inheritdoc}
     */
    public function propertyToColumnName($propertyName, $className = null)
    {
        return $this->underscore($propertyName);
    }
    /**
     * {@inheritdoc}
     */
    public function embeddedFieldToColumnName($propertyName, $embeddedColumnName, $className = null, $embeddedClassName = null)
    {
        return $this->underscore($propertyName) . '_' . $embeddedColumnName;
    }
    /**
     * {@inheritdoc}
     */
    public function referenceColumnName()
    {
        return $this->case === \CASE_UPPER ? 'ID' : 'id';
    }
    /**
     * {@inheritdoc}
     */
    public function joinColumnName($propertyName, $className = null)
    {
        return $this->underscore($propertyName) . '_' . $this->referenceColumnName();
    }
    /**
     * {@inheritdoc}
     */
    public function joinTableName($sourceEntity, $targetEntity, $propertyName = null)
    {
        return $this->classToTableName($sourceEntity) . '_' . $this->classToTableName($targetEntity);
    }
    /**
     * {@inheritdoc}
     */
    public function joinKeyColumnName($entityName, $referencedColumnName = null)
    {
        return $this->classToTableName($entityName) . '_' . ($referencedColumnName ?: $this->referenceColumnName());
    }
    /**
     * @param string $string
     *
     * @return string
     */
    private function underscore($string)
    {
        $string = \preg_replace('/(?<=[a-z])([A-Z])/', '_$1', $string);
        if ($this->case === \CASE_UPPER) {
            return \strtoupper($string);
        }
        return \strtolower($string);
    }
}
