<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\ORM\Mapping;

use MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform;
/**
 * A set of rules for determining the physical column, alias and table quotes
 *
 * @since   2.3
 * @author  Fabio B. Silva <fabio.bat.silva@gmail.com>
 */
class DefaultQuoteStrategy implements \MailPoetVendor\Doctrine\ORM\Mapping\QuoteStrategy
{
    /**
     * {@inheritdoc}
     */
    public function getColumnName($fieldName, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        return isset($class->fieldMappings[$fieldName]['quoted']) ? $platform->quoteIdentifier($class->fieldMappings[$fieldName]['columnName']) : $class->fieldMappings[$fieldName]['columnName'];
    }
    /**
     * {@inheritdoc}
     *
     * @todo Table names should be computed in DBAL depending on the platform
     */
    public function getTableName(\MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        $tableName = $class->table['name'];
        if (!empty($class->table['schema'])) {
            $tableName = $class->table['schema'] . '.' . $class->table['name'];
            if (!$platform->supportsSchemas() && $platform->canEmulateSchemas()) {
                $tableName = $class->table['schema'] . '__' . $class->table['name'];
            }
        }
        return isset($class->table['quoted']) ? $platform->quoteIdentifier($tableName) : $tableName;
    }
    /**
     * {@inheritdoc}
     */
    public function getSequenceName(array $definition, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        return isset($definition['quoted']) ? $platform->quoteIdentifier($definition['sequenceName']) : $definition['sequenceName'];
    }
    /**
     * {@inheritdoc}
     */
    public function getJoinColumnName(array $joinColumn, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        return isset($joinColumn['quoted']) ? $platform->quoteIdentifier($joinColumn['name']) : $joinColumn['name'];
    }
    /**
     * {@inheritdoc}
     */
    public function getReferencedJoinColumnName(array $joinColumn, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        return isset($joinColumn['quoted']) ? $platform->quoteIdentifier($joinColumn['referencedColumnName']) : $joinColumn['referencedColumnName'];
    }
    /**
     * {@inheritdoc}
     */
    public function getJoinTableName(array $association, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        $schema = '';
        if (isset($association['joinTable']['schema'])) {
            $schema = $association['joinTable']['schema'] . '.';
        }
        $tableName = $association['joinTable']['name'];
        if (isset($association['joinTable']['quoted'])) {
            $tableName = $platform->quoteIdentifier($tableName);
        }
        return $schema . $tableName;
    }
    /**
     * {@inheritdoc}
     */
    public function getIdentifierColumnNames(\MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform)
    {
        $quotedColumnNames = array();
        foreach ($class->identifier as $fieldName) {
            if (isset($class->fieldMappings[$fieldName])) {
                $quotedColumnNames[] = $this->getColumnName($fieldName, $class, $platform);
                continue;
            }
            // Association defined as Id field
            $joinColumns = $class->associationMappings[$fieldName]['joinColumns'];
            $assocQuotedColumnNames = \array_map(function ($joinColumn) use($platform) {
                return isset($joinColumn['quoted']) ? $platform->quoteIdentifier($joinColumn['name']) : $joinColumn['name'];
            }, $joinColumns);
            $quotedColumnNames = \array_merge($quotedColumnNames, $assocQuotedColumnNames);
        }
        return $quotedColumnNames;
    }
    /**
     * {@inheritdoc}
     */
    public function getColumnAlias($columnName, $counter, \MailPoetVendor\Doctrine\DBAL\Platforms\AbstractPlatform $platform, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class = null)
    {
        // 1 ) Concatenate column name and counter
        // 2 ) Trim the column alias to the maximum identifier length of the platform.
        //     If the alias is to long, characters are cut off from the beginning.
        // 3 ) Strip non alphanumeric characters
        // 4 ) Prefix with "_" if the result its numeric
        $columnName = $columnName . '_' . $counter;
        $columnName = \substr($columnName, -$platform->getMaxIdentifierLength());
        $columnName = \preg_replace('/[^A-Za-z0-9_]/', '', $columnName);
        $columnName = \is_numeric($columnName) ? '_' . $columnName : $columnName;
        return $platform->getSQLResultCasing($columnName);
    }
}
