<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\ORM\Event;

use MailPoetVendor\Doctrine\Common\EventArgs;
use MailPoetVendor\Doctrine\ORM\EntityManagerInterface;
use MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata;
/**
 * A method invoker based on entity lifecycle.
 *
 * @author  Fabio B. Silva <fabio.bat.silva@gmail.com>
 * @since   2.4
 */
class ListenersInvoker
{
    const INVOKE_NONE = 0;
    const INVOKE_LISTENERS = 1;
    const INVOKE_CALLBACKS = 2;
    const INVOKE_MANAGER = 4;
    /**
     * @var \MailPoetVendor\Doctrine\ORM\Mapping\EntityListenerResolver The Entity listener resolver.
     */
    private $resolver;
    /**
     * The EventManager used for dispatching events.
     *
     * @var \MailPoetVendor\Doctrine\Common\EventManager
     */
    private $eventManager;
    /**
     * Initializes a new ListenersInvoker instance.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(\MailPoetVendor\Doctrine\ORM\EntityManagerInterface $em)
    {
        $this->eventManager = $em->getEventManager();
        $this->resolver = $em->getConfiguration()->getEntityListenerResolver();
    }
    /**
     * Get the subscribed event systems
     *
     * @param \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $metadata The entity metadata.
     * @param string $eventName                             The entity lifecycle event.
     *
     * @return integer Bitmask of subscribed event systems.
     */
    public function getSubscribedSystems(\MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $metadata, $eventName)
    {
        $invoke = self::INVOKE_NONE;
        if (isset($metadata->lifecycleCallbacks[$eventName])) {
            $invoke |= self::INVOKE_CALLBACKS;
        }
        if (isset($metadata->entityListeners[$eventName])) {
            $invoke |= self::INVOKE_LISTENERS;
        }
        if ($this->eventManager->hasListeners($eventName)) {
            $invoke |= self::INVOKE_MANAGER;
        }
        return $invoke;
    }
    /**
     * Dispatches the lifecycle event of the given entity.
     *
     * @param \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $metadata The entity metadata.
     * @param string $eventName                             The entity lifecycle event.
     * @param object $entity                                The Entity on which the event occurred.
     * @param \MailPoetVendor\Doctrine\Common\EventArgs $event             The Event args.
     * @param integer $invoke                               Bitmask to invoke listeners.
     */
    public function invoke(\MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $metadata, $eventName, $entity, \MailPoetVendor\Doctrine\Common\EventArgs $event, $invoke)
    {
        if ($invoke & self::INVOKE_CALLBACKS) {
            foreach ($metadata->lifecycleCallbacks[$eventName] as $callback) {
                $entity->{$callback}($event);
            }
        }
        if ($invoke & self::INVOKE_LISTENERS) {
            foreach ($metadata->entityListeners[$eventName] as $listener) {
                $class = $listener['class'];
                $method = $listener['method'];
                $instance = $this->resolver->resolve($class);
                $instance->{$method}($entity, $event);
            }
        }
        if ($invoke & self::INVOKE_MANAGER) {
            $this->eventManager->dispatchEvent($eventName, $event);
        }
    }
}
