<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\ORM;

use MailPoetVendor\Doctrine\Common\Annotations\AnnotationReader;
use MailPoetVendor\Doctrine\Common\Annotations\AnnotationRegistry;
use MailPoetVendor\Doctrine\Common\Annotations\CachedReader;
use MailPoetVendor\Doctrine\Common\Annotations\SimpleAnnotationReader;
use MailPoetVendor\Doctrine\Common\Cache\ArrayCache;
use MailPoetVendor\Doctrine\Common\Cache\Cache as CacheDriver;
use MailPoetVendor\Doctrine\Common\Proxy\AbstractProxyFactory;
use MailPoetVendor\Doctrine\ORM\Cache\CacheConfiguration;
use MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver\MappingDriver;
use MailPoetVendor\Doctrine\ORM\Mapping\DefaultEntityListenerResolver;
use MailPoetVendor\Doctrine\ORM\Mapping\DefaultNamingStrategy;
use MailPoetVendor\Doctrine\ORM\Mapping\DefaultQuoteStrategy;
use MailPoetVendor\Doctrine\ORM\Mapping\Driver\AnnotationDriver;
use MailPoetVendor\Doctrine\ORM\Mapping\EntityListenerResolver;
use MailPoetVendor\Doctrine\ORM\Mapping\NamingStrategy;
use MailPoetVendor\Doctrine\ORM\Mapping\QuoteStrategy;
use MailPoetVendor\Doctrine\ORM\Repository\DefaultRepositoryFactory;
use MailPoetVendor\Doctrine\ORM\Repository\RepositoryFactory;
/**
 * Configuration container for all configuration options of Doctrine.
 * It combines all configuration options from DBAL & ORM.
 *
 * Internal note: When adding a new configuration option just write a getter/setter pair.
 *
 * @since 2.0
 * @author  Benjamin Eberlei <kontakt@beberlei.de>
 * @author  Guilherme Blanco <guilhermeblanco@hotmail.com>
 * @author  Jonathan Wage <jonwage@gmail.com>
 * @author  Roman Borschel <roman@code-factory.org>
 */
class Configuration extends \MailPoetVendor\Doctrine\DBAL\Configuration
{
    /**
     * Sets the directory where Doctrine generates any necessary proxy class files.
     *
     * @param string $dir
     *
     * @return void
     */
    public function setProxyDir($dir)
    {
        $this->_attributes['proxyDir'] = $dir;
    }
    /**
     * Gets the directory where Doctrine generates any necessary proxy class files.
     *
     * @return string|null
     */
    public function getProxyDir()
    {
        return isset($this->_attributes['proxyDir']) ? $this->_attributes['proxyDir'] : null;
    }
    /**
     * Gets the strategy for automatically generating proxy classes.
     *
     * @return int Possible values are constants of Doctrine\Common\Proxy\AbstractProxyFactory.
     */
    public function getAutoGenerateProxyClasses()
    {
        return isset($this->_attributes['autoGenerateProxyClasses']) ? $this->_attributes['autoGenerateProxyClasses'] : \MailPoetVendor\Doctrine\Common\Proxy\AbstractProxyFactory::AUTOGENERATE_ALWAYS;
    }
    /**
     * Sets the strategy for automatically generating proxy classes.
     *
     * @param boolean|int $autoGenerate Possible values are constants of Doctrine\Common\Proxy\AbstractProxyFactory.
     *                                  True is converted to AUTOGENERATE_ALWAYS, false to AUTOGENERATE_NEVER.
     *
     * @return void
     */
    public function setAutoGenerateProxyClasses($autoGenerate)
    {
        $this->_attributes['autoGenerateProxyClasses'] = (int) $autoGenerate;
    }
    /**
     * Gets the namespace where proxy classes reside.
     *
     * @return string|null
     */
    public function getProxyNamespace()
    {
        return isset($this->_attributes['proxyNamespace']) ? $this->_attributes['proxyNamespace'] : null;
    }
    /**
     * Sets the namespace where proxy classes reside.
     *
     * @param string $ns
     *
     * @return void
     */
    public function setProxyNamespace($ns)
    {
        $this->_attributes['proxyNamespace'] = $ns;
    }
    /**
     * Sets the cache driver implementation that is used for metadata caching.
     *
     * @param MappingDriver $driverImpl
     *
     * @return void
     *
     * @todo Force parameter to be a Closure to ensure lazy evaluation
     *       (as soon as a metadata cache is in effect, the driver never needs to initialize).
     */
    public function setMetadataDriverImpl(\MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver\MappingDriver $driverImpl)
    {
        $this->_attributes['metadataDriverImpl'] = $driverImpl;
    }
    /**
     * Adds a new default annotation driver with a correctly configured annotation reader. If $useSimpleAnnotationReader
     * is true, the notation `@Entity` will work, otherwise, the notation `@ORM\Entity` will be supported.
     *
     * @param array $paths
     * @param bool  $useSimpleAnnotationReader
     *
     * @return AnnotationDriver
     */
    public function newDefaultAnnotationDriver($paths = array(), $useSimpleAnnotationReader = \true)
    {
        \MailPoetVendor\Doctrine\Common\Annotations\AnnotationRegistry::registerFile(__DIR__ . '/Mapping/Driver/DoctrineAnnotations.php');
        if ($useSimpleAnnotationReader) {
            // Register the ORM Annotations in the AnnotationRegistry
            $reader = new \MailPoetVendor\Doctrine\Common\Annotations\SimpleAnnotationReader();
            $reader->addNamespace('MailPoetVendor\\Doctrine\\ORM\\Mapping');
            $cachedReader = new \MailPoetVendor\Doctrine\Common\Annotations\CachedReader($reader, new \MailPoetVendor\Doctrine\Common\Cache\ArrayCache());
            return new \MailPoetVendor\Doctrine\ORM\Mapping\Driver\AnnotationDriver($cachedReader, (array) $paths);
        }
        return new \MailPoetVendor\Doctrine\ORM\Mapping\Driver\AnnotationDriver(new \MailPoetVendor\Doctrine\Common\Annotations\CachedReader(new \MailPoetVendor\Doctrine\Common\Annotations\AnnotationReader(), new \MailPoetVendor\Doctrine\Common\Cache\ArrayCache()), (array) $paths);
    }
    /**
     * Adds a namespace under a certain alias.
     *
     * @param string $alias
     * @param string $namespace
     *
     * @return void
     */
    public function addEntityNamespace($alias, $namespace)
    {
        $this->_attributes['entityNamespaces'][$alias] = $namespace;
    }
    /**
     * Resolves a registered namespace alias to the full namespace.
     *
     * @param string $entityNamespaceAlias
     *
     * @return string
     *
     * @throws ORMException
     */
    public function getEntityNamespace($entityNamespaceAlias)
    {
        if (!isset($this->_attributes['entityNamespaces'][$entityNamespaceAlias])) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::unknownEntityNamespace($entityNamespaceAlias);
        }
        return \trim($this->_attributes['entityNamespaces'][$entityNamespaceAlias], '\\');
    }
    /**
     * Sets the entity alias map.
     *
     * @param array $entityNamespaces
     *
     * @return void
     */
    public function setEntityNamespaces(array $entityNamespaces)
    {
        $this->_attributes['entityNamespaces'] = $entityNamespaces;
    }
    /**
     * Retrieves the list of registered entity namespace aliases.
     *
     * @return array
     */
    public function getEntityNamespaces()
    {
        return $this->_attributes['entityNamespaces'];
    }
    /**
     * Gets the cache driver implementation that is used for the mapping metadata.
     *
     * @return MappingDriver|null
     *
     * @throws ORMException
     */
    public function getMetadataDriverImpl()
    {
        return isset($this->_attributes['metadataDriverImpl']) ? $this->_attributes['metadataDriverImpl'] : null;
    }
    /**
     * Gets the cache driver implementation that is used for the query cache (SQL cache).
     *
     * @return \MailPoetVendor\Doctrine\Common\Cache\Cache|null
     */
    public function getQueryCacheImpl()
    {
        return isset($this->_attributes['queryCacheImpl']) ? $this->_attributes['queryCacheImpl'] : null;
    }
    /**
     * Sets the cache driver implementation that is used for the query cache (SQL cache).
     *
     * @param \MailPoetVendor\Doctrine\Common\Cache\Cache $cacheImpl
     *
     * @return void
     */
    public function setQueryCacheImpl(\MailPoetVendor\Doctrine\Common\Cache\Cache $cacheImpl)
    {
        $this->_attributes['queryCacheImpl'] = $cacheImpl;
    }
    /**
     * Gets the cache driver implementation that is used for the hydration cache (SQL cache).
     *
     * @return \MailPoetVendor\Doctrine\Common\Cache\Cache|null
     */
    public function getHydrationCacheImpl()
    {
        return isset($this->_attributes['hydrationCacheImpl']) ? $this->_attributes['hydrationCacheImpl'] : null;
    }
    /**
     * Sets the cache driver implementation that is used for the hydration cache (SQL cache).
     *
     * @param \MailPoetVendor\Doctrine\Common\Cache\Cache $cacheImpl
     *
     * @return void
     */
    public function setHydrationCacheImpl(\MailPoetVendor\Doctrine\Common\Cache\Cache $cacheImpl)
    {
        $this->_attributes['hydrationCacheImpl'] = $cacheImpl;
    }
    /**
     * Gets the cache driver implementation that is used for metadata caching.
     *
     * @return \MailPoetVendor\Doctrine\Common\Cache\Cache|null
     */
    public function getMetadataCacheImpl()
    {
        return isset($this->_attributes['metadataCacheImpl']) ? $this->_attributes['metadataCacheImpl'] : null;
    }
    /**
     * Sets the cache driver implementation that is used for metadata caching.
     *
     * @param \MailPoetVendor\Doctrine\Common\Cache\Cache $cacheImpl
     *
     * @return void
     */
    public function setMetadataCacheImpl(\MailPoetVendor\Doctrine\Common\Cache\Cache $cacheImpl)
    {
        $this->_attributes['metadataCacheImpl'] = $cacheImpl;
    }
    /**
     * Adds a named DQL query to the configuration.
     *
     * @param string $name The name of the query.
     * @param string $dql  The DQL query string.
     *
     * @return void
     */
    public function addNamedQuery($name, $dql)
    {
        $this->_attributes['namedQueries'][$name] = $dql;
    }
    /**
     * Gets a previously registered named DQL query.
     *
     * @param string $name The name of the query.
     *
     * @return string The DQL query.
     *
     * @throws ORMException
     */
    public function getNamedQuery($name)
    {
        if (!isset($this->_attributes['namedQueries'][$name])) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::namedQueryNotFound($name);
        }
        return $this->_attributes['namedQueries'][$name];
    }
    /**
     * Adds a named native query to the configuration.
     *
     * @param string                 $name The name of the query.
     * @param string                 $sql  The native SQL query string.
     * @param Query\ResultSetMapping $rsm  The ResultSetMapping used for the results of the SQL query.
     *
     * @return void
     */
    public function addNamedNativeQuery($name, $sql, \MailPoetVendor\Doctrine\ORM\Query\ResultSetMapping $rsm)
    {
        $this->_attributes['namedNativeQueries'][$name] = array($sql, $rsm);
    }
    /**
     * Gets the components of a previously registered named native query.
     *
     * @param string $name The name of the query.
     *
     * @return array A tuple with the first element being the SQL string and the second
     *               element being the ResultSetMapping.
     *
     * @throws ORMException
     */
    public function getNamedNativeQuery($name)
    {
        if (!isset($this->_attributes['namedNativeQueries'][$name])) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::namedNativeQueryNotFound($name);
        }
        return $this->_attributes['namedNativeQueries'][$name];
    }
    /**
     * Ensures that this Configuration instance contains settings that are
     * suitable for a production environment.
     *
     * @return void
     *
     * @throws ORMException If a configuration setting has a value that is not
     *                      suitable for a production environment.
     */
    public function ensureProductionSettings()
    {
        $queryCacheImpl = $this->getQueryCacheImpl();
        if (!$queryCacheImpl) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::queryCacheNotConfigured();
        }
        if ($queryCacheImpl instanceof \MailPoetVendor\Doctrine\Common\Cache\ArrayCache) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::queryCacheUsesNonPersistentCache($queryCacheImpl);
        }
        $metadataCacheImpl = $this->getMetadataCacheImpl();
        if (!$metadataCacheImpl) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::metadataCacheNotConfigured();
        }
        if ($metadataCacheImpl instanceof \MailPoetVendor\Doctrine\Common\Cache\ArrayCache) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::metadataCacheUsesNonPersistentCache($metadataCacheImpl);
        }
        if ($this->getAutoGenerateProxyClasses()) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::proxyClassesAlwaysRegenerating();
        }
    }
    /**
     * Registers a custom DQL function that produces a string value.
     * Such a function can then be used in any DQL statement in any place where string
     * functions are allowed.
     *
     * DQL function names are case-insensitive.
     *
     * @param string          $name      Function name.
     * @param string|callable $className Class name or a callable that returns the function.
     *
     * @return void
     *
     * @throws ORMException
     */
    public function addCustomStringFunction($name, $className)
    {
        if (\MailPoetVendor\Doctrine\ORM\Query\Parser::isInternalFunction($name)) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::overwriteInternalDQLFunctionNotAllowed($name);
        }
        $this->_attributes['customStringFunctions'][\strtolower($name)] = $className;
    }
    /**
     * Gets the implementation class name of a registered custom string DQL function.
     *
     * @param string $name
     *
     * @return string|null
     */
    public function getCustomStringFunction($name)
    {
        $name = \strtolower($name);
        return isset($this->_attributes['customStringFunctions'][$name]) ? $this->_attributes['customStringFunctions'][$name] : null;
    }
    /**
     * Sets a map of custom DQL string functions.
     *
     * Keys must be function names and values the FQCN of the implementing class.
     * The function names will be case-insensitive in DQL.
     *
     * Any previously added string functions are discarded.
     *
     * @param array $functions The map of custom DQL string functions.
     *
     * @return void
     */
    public function setCustomStringFunctions(array $functions)
    {
        foreach ($functions as $name => $className) {
            $this->addCustomStringFunction($name, $className);
        }
    }
    /**
     * Registers a custom DQL function that produces a numeric value.
     * Such a function can then be used in any DQL statement in any place where numeric
     * functions are allowed.
     *
     * DQL function names are case-insensitive.
     *
     * @param string          $name      Function name.
     * @param string|callable $className Class name or a callable that returns the function.
     *
     * @return void
     *
     * @throws ORMException
     */
    public function addCustomNumericFunction($name, $className)
    {
        if (\MailPoetVendor\Doctrine\ORM\Query\Parser::isInternalFunction($name)) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::overwriteInternalDQLFunctionNotAllowed($name);
        }
        $this->_attributes['customNumericFunctions'][\strtolower($name)] = $className;
    }
    /**
     * Gets the implementation class name of a registered custom numeric DQL function.
     *
     * @param string $name
     *
     * @return string|null
     */
    public function getCustomNumericFunction($name)
    {
        $name = \strtolower($name);
        return isset($this->_attributes['customNumericFunctions'][$name]) ? $this->_attributes['customNumericFunctions'][$name] : null;
    }
    /**
     * Sets a map of custom DQL numeric functions.
     *
     * Keys must be function names and values the FQCN of the implementing class.
     * The function names will be case-insensitive in DQL.
     *
     * Any previously added numeric functions are discarded.
     *
     * @param array $functions The map of custom DQL numeric functions.
     *
     * @return void
     */
    public function setCustomNumericFunctions(array $functions)
    {
        foreach ($functions as $name => $className) {
            $this->addCustomNumericFunction($name, $className);
        }
    }
    /**
     * Registers a custom DQL function that produces a date/time value.
     * Such a function can then be used in any DQL statement in any place where date/time
     * functions are allowed.
     *
     * DQL function names are case-insensitive.
     *
     * @param string          $name      Function name.
     * @param string|callable $className Class name or a callable that returns the function.
     *
     * @return void
     *
     * @throws ORMException
     */
    public function addCustomDatetimeFunction($name, $className)
    {
        if (\MailPoetVendor\Doctrine\ORM\Query\Parser::isInternalFunction($name)) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::overwriteInternalDQLFunctionNotAllowed($name);
        }
        $this->_attributes['customDatetimeFunctions'][\strtolower($name)] = $className;
    }
    /**
     * Gets the implementation class name of a registered custom date/time DQL function.
     *
     * @param string $name
     *
     * @return string|null
     */
    public function getCustomDatetimeFunction($name)
    {
        $name = \strtolower($name);
        return isset($this->_attributes['customDatetimeFunctions'][$name]) ? $this->_attributes['customDatetimeFunctions'][$name] : null;
    }
    /**
     * Sets a map of custom DQL date/time functions.
     *
     * Keys must be function names and values the FQCN of the implementing class.
     * The function names will be case-insensitive in DQL.
     *
     * Any previously added date/time functions are discarded.
     *
     * @param array $functions The map of custom DQL date/time functions.
     *
     * @return void
     */
    public function setCustomDatetimeFunctions(array $functions)
    {
        foreach ($functions as $name => $className) {
            $this->addCustomDatetimeFunction($name, $className);
        }
    }
    /**
     * Sets the custom hydrator modes in one pass.
     *
     * @param array $modes An array of ($modeName => $hydrator).
     *
     * @return void
     */
    public function setCustomHydrationModes($modes)
    {
        $this->_attributes['customHydrationModes'] = array();
        foreach ($modes as $modeName => $hydrator) {
            $this->addCustomHydrationMode($modeName, $hydrator);
        }
    }
    /**
     * Gets the hydrator class for the given hydration mode name.
     *
     * @param string $modeName The hydration mode name.
     *
     * @return string|null The hydrator class name.
     */
    public function getCustomHydrationMode($modeName)
    {
        return isset($this->_attributes['customHydrationModes'][$modeName]) ? $this->_attributes['customHydrationModes'][$modeName] : null;
    }
    /**
     * Adds a custom hydration mode.
     *
     * @param string $modeName The hydration mode name.
     * @param string $hydrator The hydrator class name.
     *
     * @return void
     */
    public function addCustomHydrationMode($modeName, $hydrator)
    {
        $this->_attributes['customHydrationModes'][$modeName] = $hydrator;
    }
    /**
     * Sets a class metadata factory.
     *
     * @param string $cmfName
     *
     * @return void
     */
    public function setClassMetadataFactoryName($cmfName)
    {
        $this->_attributes['classMetadataFactoryName'] = $cmfName;
    }
    /**
     * @return string
     */
    public function getClassMetadataFactoryName()
    {
        if (!isset($this->_attributes['classMetadataFactoryName'])) {
            $this->_attributes['classMetadataFactoryName'] = 'MailPoetVendor\\Doctrine\\ORM\\Mapping\\ClassMetadataFactory';
        }
        return $this->_attributes['classMetadataFactoryName'];
    }
    /**
     * Adds a filter to the list of possible filters.
     *
     * @param string $name      The name of the filter.
     * @param string $className The class name of the filter.
     */
    public function addFilter($name, $className)
    {
        $this->_attributes['filters'][$name] = $className;
    }
    /**
     * Gets the class name for a given filter name.
     *
     * @param string $name The name of the filter.
     *
     * @return string The class name of the filter, or null of it is not
     *  defined.
     */
    public function getFilterClassName($name)
    {
        return isset($this->_attributes['filters'][$name]) ? $this->_attributes['filters'][$name] : null;
    }
    /**
     * Sets default repository class.
     *
     * @since 2.2
     *
     * @param string $className
     *
     * @return void
     *
     * @throws ORMException If not is a \MailPoetVendor\Doctrine\Common\Persistence\ObjectRepository
     */
    public function setDefaultRepositoryClassName($className)
    {
        $reflectionClass = new \ReflectionClass($className);
        if (!$reflectionClass->implementsInterface('MailPoetVendor\\Doctrine\\Common\\Persistence\\ObjectRepository')) {
            throw \MailPoetVendor\Doctrine\ORM\ORMException::invalidEntityRepository($className);
        }
        $this->_attributes['defaultRepositoryClassName'] = $className;
    }
    /**
     * Get default repository class.
     *
     * @since 2.2
     *
     * @return string
     */
    public function getDefaultRepositoryClassName()
    {
        return isset($this->_attributes['defaultRepositoryClassName']) ? $this->_attributes['defaultRepositoryClassName'] : 'MailPoetVendor\\Doctrine\\ORM\\EntityRepository';
    }
    /**
     * Sets naming strategy.
     *
     * @since 2.3
     *
     * @param NamingStrategy $namingStrategy
     *
     * @return void
     */
    public function setNamingStrategy(\MailPoetVendor\Doctrine\ORM\Mapping\NamingStrategy $namingStrategy)
    {
        $this->_attributes['namingStrategy'] = $namingStrategy;
    }
    /**
     * Gets naming strategy..
     *
     * @since 2.3
     *
     * @return NamingStrategy
     */
    public function getNamingStrategy()
    {
        if (!isset($this->_attributes['namingStrategy'])) {
            $this->_attributes['namingStrategy'] = new \MailPoetVendor\Doctrine\ORM\Mapping\DefaultNamingStrategy();
        }
        return $this->_attributes['namingStrategy'];
    }
    /**
     * Sets quote strategy.
     *
     * @since 2.3
     *
     * @param \MailPoetVendor\Doctrine\ORM\Mapping\QuoteStrategy $quoteStrategy
     *
     * @return void
     */
    public function setQuoteStrategy(\MailPoetVendor\Doctrine\ORM\Mapping\QuoteStrategy $quoteStrategy)
    {
        $this->_attributes['quoteStrategy'] = $quoteStrategy;
    }
    /**
     * Gets quote strategy.
     *
     * @since 2.3
     *
     * @return \MailPoetVendor\Doctrine\ORM\Mapping\QuoteStrategy
     */
    public function getQuoteStrategy()
    {
        if (!isset($this->_attributes['quoteStrategy'])) {
            $this->_attributes['quoteStrategy'] = new \MailPoetVendor\Doctrine\ORM\Mapping\DefaultQuoteStrategy();
        }
        return $this->_attributes['quoteStrategy'];
    }
    /**
     * Set the entity listener resolver.
     *
     * @since 2.4
     * @param \MailPoetVendor\Doctrine\ORM\Mapping\EntityListenerResolver $resolver
     */
    public function setEntityListenerResolver(\MailPoetVendor\Doctrine\ORM\Mapping\EntityListenerResolver $resolver)
    {
        $this->_attributes['entityListenerResolver'] = $resolver;
    }
    /**
     * Get the entity listener resolver.
     *
     * @since 2.4
     * @return \MailPoetVendor\Doctrine\ORM\Mapping\EntityListenerResolver
     */
    public function getEntityListenerResolver()
    {
        if (!isset($this->_attributes['entityListenerResolver'])) {
            $this->_attributes['entityListenerResolver'] = new \MailPoetVendor\Doctrine\ORM\Mapping\DefaultEntityListenerResolver();
        }
        return $this->_attributes['entityListenerResolver'];
    }
    /**
     * Set the entity repository factory.
     *
     * @since 2.4
     * @param \MailPoetVendor\Doctrine\ORM\Repository\RepositoryFactory $repositoryFactory
     */
    public function setRepositoryFactory(\MailPoetVendor\Doctrine\ORM\Repository\RepositoryFactory $repositoryFactory)
    {
        $this->_attributes['repositoryFactory'] = $repositoryFactory;
    }
    /**
     * Get the entity repository factory.
     *
     * @since 2.4
     * @return \MailPoetVendor\Doctrine\ORM\Repository\RepositoryFactory
     */
    public function getRepositoryFactory()
    {
        return isset($this->_attributes['repositoryFactory']) ? $this->_attributes['repositoryFactory'] : new \MailPoetVendor\Doctrine\ORM\Repository\DefaultRepositoryFactory();
    }
    /**
     * @since 2.5
     *
     * @return boolean
     */
    public function isSecondLevelCacheEnabled()
    {
        return isset($this->_attributes['isSecondLevelCacheEnabled']) ? $this->_attributes['isSecondLevelCacheEnabled'] : \false;
    }
    /**
     * @since 2.5
     *
     * @param boolean $flag
     *
     * @return void
     */
    public function setSecondLevelCacheEnabled($flag = \true)
    {
        $this->_attributes['isSecondLevelCacheEnabled'] = (bool) $flag;
    }
    /**
     * @since 2.5
     *
     * @param \MailPoetVendor\Doctrine\ORM\Cache\CacheConfiguration $cacheConfig
     *
     * @return void
     */
    public function setSecondLevelCacheConfiguration(\MailPoetVendor\Doctrine\ORM\Cache\CacheConfiguration $cacheConfig)
    {
        $this->_attributes['secondLevelCacheConfiguration'] = $cacheConfig;
    }
    /**
     * @since 2.5
     *
     * @return  \MailPoetVendor\Doctrine\ORM\Cache\CacheConfiguration|null
     */
    public function getSecondLevelCacheConfiguration()
    {
        if (!isset($this->_attributes['secondLevelCacheConfiguration']) && $this->isSecondLevelCacheEnabled()) {
            $this->_attributes['secondLevelCacheConfiguration'] = new \MailPoetVendor\Doctrine\ORM\Cache\CacheConfiguration();
        }
        return isset($this->_attributes['secondLevelCacheConfiguration']) ? $this->_attributes['secondLevelCacheConfiguration'] : null;
    }
    /**
     * Returns query hints, which will be applied to every query in application
     *
     * @since 2.5
     *
     * @return array
     */
    public function getDefaultQueryHints()
    {
        return isset($this->_attributes['defaultQueryHints']) ? $this->_attributes['defaultQueryHints'] : array();
    }
    /**
     * Sets array of query hints, which will be applied to every query in application
     *
     * @since 2.5
     *
     * @param array $defaultQueryHints
     */
    public function setDefaultQueryHints(array $defaultQueryHints)
    {
        $this->_attributes['defaultQueryHints'] = $defaultQueryHints;
    }
    /**
     * Gets the value of a default query hint. If the hint name is not recognized, FALSE is returned.
     *
     * @since 2.5
     *
     * @param string $name The name of the hint.
     *
     * @return mixed The value of the hint or FALSE, if the hint name is not recognized.
     */
    public function getDefaultQueryHint($name)
    {
        return isset($this->_attributes['defaultQueryHints'][$name]) ? $this->_attributes['defaultQueryHints'][$name] : \false;
    }
    /**
     * Sets a default query hint. If the hint name is not recognized, it is silently ignored.
     *
     * @since 2.5
     *
     * @param string $name  The name of the hint.
     * @param mixed  $value The value of the hint.
     */
    public function setDefaultQueryHint($name, $value)
    {
        $this->_attributes['defaultQueryHints'][$name] = $value;
    }
}
