<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver;

use MailPoetVendor\Doctrine\Common\Persistence\Mapping\MappingException;
/**
 * Base driver for file-based metadata drivers.
 *
 * A file driver operates in a mode where it loads the mapping files of individual
 * classes on demand. This requires the user to adhere to the convention of 1 mapping
 * file per class and the file names of the mapping files must correspond to the full
 * class name, including namespace, with the namespace delimiters '\', replaced by dots '.'.
 *
 * @link   www.doctrine-project.org
 * @since  2.2
 * @author Benjamin Eberlei <kontakt@beberlei.de>
 * @author Guilherme Blanco <guilhermeblanco@hotmail.com>
 * @author Jonathan H. Wage <jonwage@gmail.com>
 * @author Roman Borschel <roman@code-factory.org>
 */
abstract class FileDriver implements \MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver\MappingDriver
{
    /**
     * @var FileLocator
     */
    protected $locator;
    /**
     * @var array|null
     */
    protected $classCache;
    /**
     * @var string|null
     */
    protected $globalBasename;
    /**
     * Initializes a new FileDriver that looks in the given path(s) for mapping
     * documents and operates in the specified operating mode.
     *
     * @param string|array|FileLocator $locator       A FileLocator or one/multiple paths
     *                                                where mapping documents can be found.
     * @param string|null              $fileExtension
     */
    public function __construct($locator, $fileExtension = null)
    {
        if ($locator instanceof \MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver\FileLocator) {
            $this->locator = $locator;
        } else {
            $this->locator = new \MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver\DefaultFileLocator((array) $locator, $fileExtension);
        }
    }
    /**
     * Sets the global basename.
     *
     * @param string $file
     *
     * @return void
     */
    public function setGlobalBasename($file)
    {
        $this->globalBasename = $file;
    }
    /**
     * Retrieves the global basename.
     *
     * @return string|null
     */
    public function getGlobalBasename()
    {
        return $this->globalBasename;
    }
    /**
     * Gets the element of schema meta data for the class from the mapping file.
     * This will lazily load the mapping file if it is not loaded yet.
     *
     * @param string $className
     *
     * @return array The element of schema meta data.
     *
     * @throws MappingException
     */
    public function getElement($className)
    {
        if ($this->classCache === null) {
            $this->initialize();
        }
        if (isset($this->classCache[$className])) {
            return $this->classCache[$className];
        }
        $result = $this->loadMappingFile($this->locator->findMappingFile($className));
        if (!isset($result[$className])) {
            throw \MailPoetVendor\Doctrine\Common\Persistence\Mapping\MappingException::invalidMappingFile($className, \str_replace('\\', '.', $className) . $this->locator->getFileExtension());
        }
        $this->classCache[$className] = $result[$className];
        return $result[$className];
    }
    /**
     * {@inheritDoc}
     */
    public function isTransient($className)
    {
        if ($this->classCache === null) {
            $this->initialize();
        }
        if (isset($this->classCache[$className])) {
            return \false;
        }
        return !$this->locator->fileExists($className);
    }
    /**
     * {@inheritDoc}
     */
    public function getAllClassNames()
    {
        if ($this->classCache === null) {
            $this->initialize();
        }
        if (!$this->classCache) {
            return (array) $this->locator->getAllClassNames($this->globalBasename);
        }
        return \array_merge(\array_keys($this->classCache), (array) $this->locator->getAllClassNames($this->globalBasename));
    }
    /**
     * Loads a mapping file with the given name and returns a map
     * from class/entity names to their corresponding file driver elements.
     *
     * @param string $file The mapping file to load.
     *
     * @return array
     */
    protected abstract function loadMappingFile($file);
    /**
     * Initializes the class cache from all the global files.
     *
     * Using this feature adds a substantial performance hit to file drivers as
     * more metadata has to be loaded into memory than might actually be
     * necessary. This may not be relevant to scenarios where caching of
     * metadata is in place, however hits very hard in scenarios where no
     * caching is used.
     *
     * @return void
     */
    protected function initialize()
    {
        $this->classCache = [];
        if (null !== $this->globalBasename) {
            foreach ($this->locator->getPaths() as $path) {
                $file = $path . '/' . $this->globalBasename . $this->locator->getFileExtension();
                if (\is_file($file)) {
                    $this->classCache = \array_merge($this->classCache, $this->loadMappingFile($file));
                }
            }
        }
    }
    /**
     * Retrieves the locator used to discover mapping files by className.
     *
     * @return FileLocator
     */
    public function getLocator()
    {
        return $this->locator;
    }
    /**
     * Sets the locator used to discover mapping files by className.
     *
     * @param FileLocator $locator
     */
    public function setLocator(\MailPoetVendor\Doctrine\Common\Persistence\Mapping\Driver\FileLocator $locator)
    {
        $this->locator = $locator;
    }
}
