<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\Common\Collections;

use ArrayIterator;
use Closure;
use MailPoetVendor\Doctrine\Common\Collections\Expr\ClosureExpressionVisitor;
/**
 * An ArrayCollection is a Collection implementation that wraps a regular PHP array.
 *
 * Warning: Using (un-)serialize() on a collection is not a supported use-case
 * and may break when we change the internals in the future. If you need to
 * serialize a collection use {@link toArray()} and reconstruct the collection
 * manually.
 *
 * @since  2.0
 * @author Guilherme Blanco <guilhermeblanco@hotmail.com>
 * @author Jonathan Wage <jonwage@gmail.com>
 * @author Roman Borschel <roman@code-factory.org>
 */
class ArrayCollection implements \MailPoetVendor\Doctrine\Common\Collections\Collection, \MailPoetVendor\Doctrine\Common\Collections\Selectable
{
    /**
     * An array containing the entries of this collection.
     *
     * @var array
     */
    private $elements;
    /**
     * Initializes a new ArrayCollection.
     *
     * @param array $elements
     */
    public function __construct(array $elements = array())
    {
        $this->elements = $elements;
    }
    /**
     * Creates a new instance from the specified elements.
     *
     * This method is provided for derived classes to specify how a new
     * instance should be created when constructor semantics have changed.
     *
     * @param array $elements Elements.
     *
     * @return static
     */
    protected function createFrom(array $elements)
    {
        return new static($elements);
    }
    /**
     * {@inheritDoc}
     */
    public function toArray()
    {
        return $this->elements;
    }
    /**
     * {@inheritDoc}
     */
    public function first()
    {
        return \reset($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function last()
    {
        return \end($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function key()
    {
        return \key($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function next()
    {
        return \next($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function current()
    {
        return \current($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function remove($key)
    {
        if (!isset($this->elements[$key]) && !\array_key_exists($key, $this->elements)) {
            return null;
        }
        $removed = $this->elements[$key];
        unset($this->elements[$key]);
        return $removed;
    }
    /**
     * {@inheritDoc}
     */
    public function removeElement($element)
    {
        $key = \array_search($element, $this->elements, \true);
        if ($key === \false) {
            return \false;
        }
        unset($this->elements[$key]);
        return \true;
    }
    /**
     * Required by interface ArrayAccess.
     *
     * {@inheritDoc}
     */
    public function offsetExists($offset)
    {
        return $this->containsKey($offset);
    }
    /**
     * Required by interface ArrayAccess.
     *
     * {@inheritDoc}
     */
    public function offsetGet($offset)
    {
        return $this->get($offset);
    }
    /**
     * Required by interface ArrayAccess.
     *
     * {@inheritDoc}
     */
    public function offsetSet($offset, $value)
    {
        if (!isset($offset)) {
            return $this->add($value);
        }
        $this->set($offset, $value);
    }
    /**
     * Required by interface ArrayAccess.
     *
     * {@inheritDoc}
     */
    public function offsetUnset($offset)
    {
        return $this->remove($offset);
    }
    /**
     * {@inheritDoc}
     */
    public function containsKey($key)
    {
        return isset($this->elements[$key]) || \array_key_exists($key, $this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function contains($element)
    {
        return \in_array($element, $this->elements, \true);
    }
    /**
     * {@inheritDoc}
     */
    public function exists(\Closure $p)
    {
        foreach ($this->elements as $key => $element) {
            if ($p($key, $element)) {
                return \true;
            }
        }
        return \false;
    }
    /**
     * {@inheritDoc}
     */
    public function indexOf($element)
    {
        return \array_search($element, $this->elements, \true);
    }
    /**
     * {@inheritDoc}
     */
    public function get($key)
    {
        return isset($this->elements[$key]) ? $this->elements[$key] : null;
    }
    /**
     * {@inheritDoc}
     */
    public function getKeys()
    {
        return \array_keys($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function getValues()
    {
        return \array_values($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function count()
    {
        return \count($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function set($key, $value)
    {
        $this->elements[$key] = $value;
    }
    /**
     * {@inheritDoc}
     */
    public function add($element)
    {
        $this->elements[] = $element;
        return \true;
    }
    /**
     * {@inheritDoc}
     */
    public function isEmpty()
    {
        return empty($this->elements);
    }
    /**
     * Required by interface IteratorAggregate.
     *
     * {@inheritDoc}
     */
    public function getIterator()
    {
        return new \ArrayIterator($this->elements);
    }
    /**
     * {@inheritDoc}
     */
    public function map(\Closure $func)
    {
        return $this->createFrom(\array_map($func, $this->elements));
    }
    /**
     * {@inheritDoc}
     */
    public function filter(\Closure $p)
    {
        return $this->createFrom(\array_filter($this->elements, $p));
    }
    /**
     * {@inheritDoc}
     */
    public function forAll(\Closure $p)
    {
        foreach ($this->elements as $key => $element) {
            if (!$p($key, $element)) {
                return \false;
            }
        }
        return \true;
    }
    /**
     * {@inheritDoc}
     */
    public function partition(\Closure $p)
    {
        $matches = $noMatches = array();
        foreach ($this->elements as $key => $element) {
            if ($p($key, $element)) {
                $matches[$key] = $element;
            } else {
                $noMatches[$key] = $element;
            }
        }
        return array($this->createFrom($matches), $this->createFrom($noMatches));
    }
    /**
     * Returns a string representation of this object.
     *
     * @return string
     */
    public function __toString()
    {
        return __CLASS__ . '@' . \spl_object_hash($this);
    }
    /**
     * {@inheritDoc}
     */
    public function clear()
    {
        $this->elements = array();
    }
    /**
     * {@inheritDoc}
     */
    public function slice($offset, $length = null)
    {
        return \array_slice($this->elements, $offset, $length, \true);
    }
    /**
     * {@inheritDoc}
     */
    public function matching(\MailPoetVendor\Doctrine\Common\Collections\Criteria $criteria)
    {
        $expr = $criteria->getWhereExpression();
        $filtered = $this->elements;
        if ($expr) {
            $visitor = new \MailPoetVendor\Doctrine\Common\Collections\Expr\ClosureExpressionVisitor();
            $filter = $visitor->dispatch($expr);
            $filtered = \array_filter($filtered, $filter);
        }
        if ($orderings = $criteria->getOrderings()) {
            $next = null;
            foreach (\array_reverse($orderings) as $field => $ordering) {
                $next = \MailPoetVendor\Doctrine\Common\Collections\Expr\ClosureExpressionVisitor::sortByField($field, $ordering == \MailPoetVendor\Doctrine\Common\Collections\Criteria::DESC ? -1 : 1, $next);
            }
            \uasort($filtered, $next);
        }
        $offset = $criteria->getFirstResult();
        $length = $criteria->getMaxResults();
        if ($offset || $length) {
            $filtered = \array_slice($filtered, (int) $offset, $length);
        }
        return $this->createFrom($filtered);
    }
}
